<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

include 'header.php';

$db = new DB();
$conn = $db->getConnection();

// Get filter inputs from GET request
$customerNameFilter = $_GET['customer_name'] ?? '';
$startDate = $_GET['start_date'] ?? '';
$endDate = $_GET['end_date'] ?? '';
$dueOnly = isset($_GET['due_only']) && $_GET['due_only'] == '1'; // New filter

// Prepare WHERE conditions and params
$where = [];
$params = [];
$types = '';

if (!empty($customerNameFilter)) {
    $where[] = "c.name LIKE ?";
    $params[] = "%" . $customerNameFilter . "%";
    $types .= 's';
}

if (!empty($startDate) && !empty($endDate)) {
    $where[] = "s.date_added BETWEEN ? AND ?";
    $params[] = $startDate . " 00:00:00";
    $params[] = $endDate . " 23:59:59";
    $types .= 'ss';
} elseif (!empty($startDate)) {
    $where[] = "s.date_added >= ?";
    $params[] = $startDate . " 00:00:00";
    $types .= 's';
} elseif (!empty($endDate)) {
    $where[] = "s.date_added <= ?";
    $params[] = $endDate . " 23:59:59";
    $types .= 's';
}

// Base query with pay_amount and balance
$sql = "
    SELECT 
        s.id, 
        s.date_added, 
        s.pay_amount,
        c.name AS customer_name,
        s.transport_charge,
        (SELECT SUM(price * qty) FROM sales_items_sold WHERE sales_id = s.id) AS total_amount
    FROM sales s
    LEFT JOIN customers c ON s.customer_id = c.id_number
";

// Apply WHERE clauses
if (!empty($where)) {
    $sql .= " WHERE " . implode(" AND ", $where);
}

// Add due balance filter
if ($dueOnly) {
    $sql .= (!empty($where) ? " AND " : " WHERE ") . "(SELECT SUM(price * qty) FROM sales_items_sold WHERE sales_id = s.id) > s.pay_amount";
}

$sql .= " ORDER BY s.date_added DESC";

$stmt = $conn->prepare($sql);

if ($params) {
    $stmt->bind_param($types, ...$params);
}

$stmt->execute();
$result = $stmt->get_result();
?>

<div class="content-wrapper">
  <div class="container-xxl flex-grow-1 container-p-y">
    <h4>Sales List</h4>

    <div class="card">
      <div class="card-header">
        <!-- Filter Form -->
        <form method="GET" action="" class="row g-3 mb-4">
          <div class="col-md-3">
            <input
              type="text"
              name="customer_name"
              class="form-control"
              placeholder="Filter by Customer Name"
              value="<?= htmlspecialchars($customerNameFilter) ?>"
            >
          </div>
          <div class="col-md-2">
            <input
              type="date"
              name="start_date"
              class="form-control"
              placeholder="Start Date"
              value="<?= htmlspecialchars($startDate) ?>"
            >
          </div>
          <div class="col-md-2">
            <input
              type="date"
              name="end_date"
              class="form-control"
              placeholder="End Date"
              value="<?= htmlspecialchars($endDate) ?>"
            >
          </div>
          <div class="col-md-3">
            <div class="form-check mt-2">
              <input class="form-check-input" type="checkbox" name="due_only" value="1" id="dueOnly" <?= $dueOnly ? 'checked' : '' ?>>
              <label class="form-check-label" for="dueOnly">
                Show Due Balance Only
              </label>
            </div>
          </div>
          <div class="col-md-2">
            <button type="submit" class="btn btn-primary w-100">Filter</button>
          </div>
        </form>
      </div>
    </div>

    <div class="card" style="margin-top:2%;">
      <div class="table-responsive">
        <table class="table table-striped">
          <thead>
            <tr>
              <th>Sale ID</th>
              <th>Customer Name</th>
              <th>Sale Date</th>
              <th>Item Total</th>
              <th>Transport Amount (LKR)</th>
              <th>Net Total</th>
              <th>Paid (LKR)</th>
              <th>Balance (LKR)</th>
              <th>Actions</th>
            </tr>
          </thead>
          <tbody>
            <?php if ($result->num_rows === 0): ?>
              <tr>
                <td colspan="7">No sales found.</td>
              </tr>
            <?php else: ?>
              <?php while ($sale = $result->fetch_assoc()): 
                $total = $sale['total_amount'] ?? 0;
                $paid = $sale['pay_amount'] ?? 0;
                $trasport_total = $sale['transport_charge'];
                $net_total = $total+$trasport_total;
                $balance = $net_total - $paid;
                
                
              ?>
                <tr>
                  <td><?= htmlspecialchars($sale['id']) ?></td>
                  <td><?= htmlspecialchars($sale['customer_name'] ?? 'N/A') ?></td>
                  <td><?= htmlspecialchars(date('Y-m-d', strtotime($sale['date_added']))) ?></td>
                   <td><?= number_format($total, 2) ?></td>
                  <td><?= number_format($trasport_total, 2) ?></td>
                 
                  <td><?= number_format($net_total, 2) ?></td>
                  <td><?= number_format($paid, 2) ?></td>
                  <td><?= number_format($balance, 2) ?></td>
                  <td>
                    <a href="view_sale.php?id=<?= $sale['id'] ?>" class="btn btn-sm btn-primary">View</a>
                    <a href="print_sale.php?id=<?= $sale['id'] ?>" class="btn btn-sm btn-success">Print</a>
                    <?php if ($userRole != "Staff") {
                      ?><a href="delete_sale.php?id=<?= $sale['id'] ?>" class="btn btn-sm btn-danger" onclick="return confirm('Are you sure you want to delete this sale?');">Delete</a><?php } ?>
                  </td>
                </tr>
              <?php endwhile; ?>
            <?php endif; ?>
          </tbody>
        </table>
      </div>
    </div>
  </div>
</div>

<?php include 'footer.php'; ?>
